# -*- coding: utf-8 -*-
"""
Created on Thursday 02 Nov 2023
@author: bane
"""

from surprise import AlgoBase
from surprise import PredictionImpossible
from surprise import NormalPredictor, BaselineOnly

import math
import numpy as np
import heapq
import pandas as pd
import random

from RecommenderMetrics import RecommenderMetrics
from YahooDataset import YahooDataset


class ContentKNNAlgorithm(AlgoBase):

    def __init__(self, k=40, sim_options={}):
        AlgoBase.__init__(self)
        self.k = k
        self.similarities = np.zeros(5)

    def fit(self, trainset):
        trainset.rating_scale = (1, 5)  # 1 to 13
        AlgoBase.fit(self, trainset)

        print("Computing content-based similarity matrix...")

        # yd = YahooDataset()
        genres = yd.getGenres()
        years = yd.getYears()

        # Compute genre distance for every movie combination as a 2x2 matrix
        self.similarities = np.zeros((self.trainset.n_items, self.trainset.n_items))

        for thisRating in range(self.trainset.n_items):
            if thisRating % 100 == 0:
                print(thisRating, " of ", self.trainset.n_items)

            for otherRating in range(thisRating + 1, self.trainset.n_items):
                thisMovieID = int(self.trainset.to_raw_iid(thisRating))
                otherMovieID = int(self.trainset.to_raw_iid(otherRating))
                genreSimilarity = self.computeGenreSimilarity(thisMovieID, otherMovieID, genres)
                yearSimilarity = self.computeYearSimilarity(thisMovieID, otherMovieID, years)

                self.similarities[thisRating, otherRating] = genreSimilarity * yearSimilarity
                self.similarities[otherRating, thisRating] = self.similarities[thisRating, otherRating]

        print("...done.")

        return self

    def computeGenreSimilarity(self, movie1, movie2, genres):
        genres1 = genres[movie1]
        genres2 = genres[movie2]
        sumxx, sumxy, sumyy = 0, 0, 0
        for i in range(len(genres1)):
            x = genres1[i]
            y = genres2[i]
            sumxx += x * x
            sumyy += y * y
            sumxy += x * y

        if sumxx == 0 | sumyy == 0:
            return 0

        return sumxy / math.sqrt(sumxx * sumyy)

    def computeYearSimilarity(self, movie1, movie2, years):
        if years[movie1] == 0 or years[movie2] == 0:
            return 0

        diff = abs(years[movie1] - years[movie2])
        sim = math.exp(-diff / 10.0)
        return sim

    def estimate(self, u, i):

        if not (self.trainset.knows_user(u) and self.trainset.knows_item(i)):
            raise PredictionImpossible('User and/or item is unkown.')

        # Build up similarity scores between this item and everything the user rated
        neighbors = []
        for rating in self.trainset.ur[u]:
            similarity = self.similarities[i, rating[0]]
            neighbors.append((similarity, rating[1]))

        # Extract the top-K most-similar ratings
        k_neighbors = heapq.nlargest(self.k, neighbors, key=lambda t: t[0])

        # Compute average sim score of K neighbors weighted by user ratings
        simTotal = weightedSum = 0
        for (simScore, rating) in k_neighbors:
            if (simScore > 0):
                simTotal += simScore
                weightedSum += simScore * rating

        if (simTotal == 0):
            raise PredictionImpossible('No neighbors')

        predictedRating = weightedSum / simTotal

        return predictedRating


if __name__ == "__main__":
    np.random.seed(0)
    random.seed(0)

    yd = YahooDataset()
    algo = ContentKNNAlgorithm()
    rm = RecommenderMetrics()
    
    # train 
    train_df = yd.loadYahooPandasTrainingDataFrame()
    trainSet = yd.loadFromPandas(train_df).build_full_trainset()
    # print(trainSet.n_items)
    algo.fit(trainSet)
    
    # test
    test_df = yd.loadYahooPandasTestDataFrame()
    testSet = yd.loadFromPandas(test_df)
    testSet = testSet.build_full_trainset()
    testSet = testSet.build_testset()

    predictions = algo.test(testSet)
    RMSE = rm.RMSE(predictions)
    print(RMSE)
